<?php
// JwtUtil.php - Classe utilitária simples para JWT (JSON Web Token)
// NOTA: Em um ambiente de produção real, é altamente recomendado usar uma biblioteca JWT madura como firebase/php-jwt.
// Esta implementação é simplificada para o ambiente de hospedagem compartilhada sem Composer.

require_once 'config.php';

class JwtUtil {
    
    private static function base64url_encode($data) {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    private static function base64url_decode($data) {
        return base64_decode(strtr($data, '-_', '+/'));
    }

    public static function generateToken($payload) {
        $header = [
            'typ' => 'JWT',
            'alg' => 'HS256'
        ];

        $payload['iat'] = time();
        $payload['exp'] = time() + TOKEN_EXPIRATION; // Definido em config.php

        $base64UrlHeader = self::base64url_encode(json_encode($header));
        $base64UrlPayload = self::base64url_encode(json_encode($payload));

        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, SECRET_KEY, true);
        $base64UrlSignature = self::base64url_encode($signature);

        return $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;
    }

    public static function validateToken($jwt) {
        $parts = explode('.', $jwt);
        if (count($parts) !== 3) {
            return false;
        }

        list($base64UrlHeader, $base64UrlPayload, $base64UrlSignature) = $parts;

        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, SECRET_KEY, true);
        $base64UrlExpectedSignature = self::base64url_encode($signature);

        if ($base64UrlSignature !== $base64UrlExpectedSignature) {
            return false; // Assinatura inválida
        }

        $payload = json_decode(self::base64url_decode($base64UrlPayload), true);

        if (isset($payload['exp']) && $payload['exp'] < time()) {
            return false; // Token expirado
        }

        return $payload;
    }
}
?>
