<?php
// jwt.php - JWT HS256 minimalista
require_once __DIR__ . '/config.php';

function b64url_encode($data) { return rtrim(strtr(base64_encode($data), '+/', '-_'), '='); }
function b64url_decode($data) { return base64_decode(strtr($data, '-_', '+/')); }

function jwt_encode(array $payload) {
  $header = ['alg' => 'HS256', 'typ' => 'JWT'];
  $segments = [ b64url_encode(json_encode($header)), b64url_encode(json_encode($payload)) ];
  $signing_input = implode('.', $segments);
  $signature = hash_hmac('sha256', $signing_input, JWT_SECRET, true);
  $segments[] = b64url_encode($signature);
  return implode('.', $segments);
}

function jwt_decode(string $jwt) {
  $parts = explode('.', $jwt);
  if (count($parts) !== 3) return [null, 'invalid'];
  list($h64, $p64, $s64) = $parts;
  $header = json_decode(b64url_decode($h64), true);
  $payload = json_decode(b64url_decode($p64), true);
  $sig = b64url_decode($s64);
  $valid = hash_equals($sig, hash_hmac('sha256', "$h64.$p64", JWT_SECRET, true));
  if (!$valid) return [null, 'signature'];
  if (isset($payload['exp']) && time() >= $payload['exp']) return [null, 'expired'];
  return [$payload, null];
}

function jwt_new_for_user($user) {
  $now = time();
  $payload = [
    'iss' => JWT_ISS,
    'iat' => $now,
    'exp' => $now + JWT_EXP_SECONDS,
    'sub' => (string)$user['id'],
    'name' => $user['nome'] ?? null,
    'email' => $user['email'] ?? null
  ];
  return jwt_encode($payload);
}
